# OraAI语义模型开发规范

## 适用场景
当您没有使用dbt项目时，需要通过手动方式构建 OraAI项目，此时无法使用`orai-cli init`或`orai-cli generate`等依赖 dbt 元数据的命令。

## 手动接入步骤

### 1. 连接数据源
- 访问 OraAI控制台
- 根据向导完成数据源配置（支持 starRocks等主流数据仓库）

### 2. 创建项目结构
推荐目录结构示例：
```text
ora-project/
├── ora.yml                 # 主配置文件
├── marketing/              # 业务域模块
│   └── semantic_models/    # 语义模型目录
│       ├── customers.yml
│       └── campaigns.yml
└── finance/
    └── semantic_models/
        ├── transactions.yml
        └── accounts.yml
```

### 3. 配置 ora.yml
```yaml
projects:
  - path: ./marketing
    data_source_name: data_warehouse_prod  # 对应UI配置的数据源名称
    schema: marketing_analytics
    database: enterprise_db
    semantic_model_paths:
      - semantic_models/  # 模型文件相对路径
  
  - path: ./finance
    data_source_name: data_warehouse_prod
    schema: finance
    database: enterprise_db 
    semantic_model_paths:
      - semantic_models/
```

## 语义模型开发规范

### 1. 客户模型示例
```yaml
# marketing/semantic_models/customers.yml
name: customers
description: 营销域客户主数据
dimensions:
  - name: customer_id
    description: 客户唯一标识
    type: VARCHAR
    searchable: true  # 启用自然语言搜索
    
  - name: member_level  
    description: 会员等级(S1-S6)
    type: VARCHAR
    options: ["S1","S2","S3","S4","S5","S6"]

relationships:
  - name: orders       # 关联订单模型
    source_col: customer_id
    ref_col: buyer_id
    cardinality: one-to-many
```

### 2. 交易模型示例
```yaml
# finance/semantic_models/transactions.yml  
name: transactions
description: 财务交易事实表
measures:
  - name: amount
    description: 交易金额(含税)
    type: DECIMAL

metrics:
  - name: avg_transaction
    expr: "SUM(amount)/COUNT(transaction_id)"
    description: 笔均交易金额
    args:
      - name: date_range
        type: string
        description: 统计时间范围
```

## 部署流程(实践案例详见入门指南-快速部署（发布模型）菜单)

### 1. 认证
```bash
$ orai-cli auth
```

### 2. 验证模型配置
```bash
$ orai-cli parse
```

### 3. 部署到OraAI平台
```bash
$ orai-cli deploy
```

## 最佳实践

### 1. 模型设计原则：
- 按业务域划分模型文件
- 保持字段名与数据库一致
- 为关键字段添加 searchable: true

### 2. 版本控制：
- 将配置文件纳入Git管理
- 使用语义化版本控制模型变更

### 3. 增量开发：
- 先完成核心模型部署
- 逐步添加指标和关联关系

### 4. 文档规范：
- 所有模型/字段必须包含描述
- 枚举值需明确业务含义
- 复杂指标需注明计算公式

该手动接入方案虽然需要更多前期配置，但可以灵活支持各类数据架构，特别适合尚未采用dbt的传统数据仓库环境。

---

# ora.yml 配置文件指南

## ora.yml文件概述

ora.yml 是 OraAI 项目的主配置文件，用于定义数据源连接和语义模型位置。

## 配置结构

采用 projects 列表结构，支持多项目配置。

## 核心字段说明

| 字段 | 必填 | 说明 |
|------|:----:|------|
| projects | 是 | 项目配置列表 |
| projects[].path | 是 | 项目路径（相对于ora.yml所在目录，.表示根目录） |
| projects[].data_source_name | 是 | OraAI中定义的数据源连接名称 |
| projects[].schema | 是 | 数据库schema名称 |
| projects[].database | 是 | 数据库名称 |
| projects[].model_paths | 是 | 模型文件目录路径列表（相对路径） |
| projects[].semantic_model_paths | 是 | 语义模型定义文件目录路径列表（相对路径） |

## 配置示例

### 多项目配置示例
```yaml
projects:
  - path: ./marketing
    data_source_name: starrocks_marketing   # 营销数据源
    schema: marketing_analytics
    database: marketing_data
    model_paths:
      - models/                             # 模型文件目录
    semantic_model_paths:
      - semantic_models/                    # 语义模型目录
  
  - path: ./finance
    data_source_name: postgres_finance      # 财务数据源
    schema: finance
    database: financial_data
    model_paths:
      - models/
      - legacy_models/                      # 多目录配置
    semantic_model_paths:
      - semantic_models/
```

### 单项目精简配置
```yaml
projects:
  - path: .                                 # 当前目录
    data_source_name: starrocks_prod
    schema: analytics
    database: business_data
    model_paths:
      - models/
    semantic_model_paths:
      - semantic_models/
```

## 最佳实践

### 1. 路径规范

- 使用相对路径时以/结尾
- 根目录配置使用 . 表示

### 2. 多环境管理
```yaml
# 开发环境
projects:
  - path: .
    data_source_name: starrocks_dev
    schema: analytics_dev
```

### 3. 目录组织建议
```text
project-root/
├── ora.yml
├── models/
│   ├── marketing.yml
│   └── sales.yml
└── semantic_models/
    ├── dimensions/
    └── metrics/
```

### 4. 版本控制
- 将ora.yml纳入版本控制
- 敏感信息（如密码）应通过环境变量配置

该配置文件作为OraAI项目的核心枢纽，合理配置可确保数据源连接和模型定位的准确性。

--- 

# OraAI语义模型规范指南

## 模型对象结构

### 1. 根模型配置(Model Object)
```yaml
name: 订单模型                                  # 必填
description: "电商平台订单数据主模型"              # 必填
data_source_name: mysql_erp                    # 可选，默认继承ora.yml
dimensions:                                    # 分析维度
  - name: order_date
    type: date
measures:                                       # 数值度量
  - name: amount
    type: decimal
relationships:  # 关联关系
  - name: customers
    source_col: customer_id
    ref_col: user_id
```

**关键字段说明：**
- 必填字段：name, description
- 可继承字段：data_source_name, database, schema
- 组件类型：dimensions/measures/metrics/filters/relationships

## 核心组件规范

### 2. 维度(Dimension)

```yaml
dimensions:
  - name: payment_method                # 必填
description: "支付方式(微信/支付宝/银行卡)" # 必填
type: string                            # 必填
searchable: true                        # 可选，默认false
options:                                # 枚举值示例
      - "微信支付"
      - "支付宝"
      - "银联"
```

**数据类型支持：**
- 基础类型：string, int, decimal
- 时间类型：date, timestamp
- 布尔类型：boolean

### 3. 度量(Measure)

```yaml
measures:
  - name: total_amount                  # 必填
description: "订单总金额(含税)"           # 必填
type: decimal                           # 原始字段类型
```

> 注意：type表示底层数据类型，非聚合方式

### 4. 业务指标(Metric)

```yaml
metrics:
  - name: conversion_rate  # 必填
    expr: "count(order_id)/count(distinct session_id)"  # 计算表达式
    description: "访客转化率"  # 必填
    args:  # 动态参数
      - name: date_range
        type: string
        description: "统计周期"
```

### 5. 过滤器(Filter)

```yaml
filters:
  - name: high_value_orders  # 必填
    expr: "amount > {{min_amount}}"  # 条件表达式
    description: "高价值订单筛选"  # 必填
    args:
      - name: min_amount
        type: decimal
        default: 1000
```

## 关联关系配置

### 6. 模型关联(Relationship)

```yaml
relationships:
  - name: customers  # 关联模型名
    source_col: customer_id  # 本模型关联字段
    ref_col: id       # 目标模型关联字段
    type: LEFT        # 连接类型
    cardinality: many-to-one  # 关系基数
    description: "关联客户主表"  # 必填
```

**连接类型：**
- LEFT/INNER/RIGHT/FULL OUTER

**基数类型：**
- one-to-one/one-to-many/many-to-one/many-to-many

## 完整示例

### 电商模型示例
```yaml
name: orders
description: "电商订单核心事实表"
dimensions:
  - name: order_id
    type: string
    description: "订单唯一标识"
    
  - name: order_date
    type: date
    description: "订单创建日期"

measures:
  - name: amount
    type: decimal
    description: "订单金额(元)"

metrics:
  - name: avg_order_value
    expr: "sum(amount)/count(distinct order_id)"
    description: "客单价"

relationships:
  - name: customers
    source_col: user_id
    ref_col: id
    cardinality: many-to-one
    description: "关联用户维度表"
```

## 最佳实践

### 1. 命名规范
- 使用snake_case命名法
- 保持字段名与数据库一致

### 2. 文档质量
- 描述需包含业务含义和计算逻辑
- 枚举值需明确业务选项

### 3. 关系设计
- 确保source_col/ref_col类型匹配
- 多对多关系需通过中间表实现

### 4. 版本控制
- 模型文件按业务域拆分
- 重大变更需保留版本历史

该规范为构建可解释、可维护的语义层提供了标准化框架，建议结合具体业务需求进行定制化扩展。

---


# 维度与度量

## 语义层字段属性定义

### 维度与度量概述

维度和度量是语义模型的核心构建元素，它们定义了可用于分析的字段，描述数据特征并支持各类计算。

- **维度**：描述性属性（如产品类别、日期或用户ID），用于数据分组、筛选和切片
- **度量**：可量化的数值（如销售额、数量或成本），用于聚合计算和指标构建

定义维度和度量的基本结构如下：

```yaml
dimensions:
  - name: dimension_name  # 必需 - 唯一标识符，通常基于列名
    description: "业务定义" # 必需 - 清晰明确的业务解释
    type: data_type       # 必需 - 基础数据类型（如string, date）

measures:
  - name: measure_name    # 必需 - 唯一标识符，通常基于列名
    description: "业务定义" # 必需 - 清晰明确的业务解释
    type: data_type       # 必需 - 基础数据类型（如number, integer）
```

## 维度

维度是数据的描述性属性或特征，通常为非数字或分类字段，用于切片、分组和筛选数据。可以将维度视为分析中回答"谁、什么、哪里、何时、为什么"等问题的要素。常见示例包括日期、用户ID、产品类别、地理位置或状态标识。

| 字段 | 是否必需 | 描述 |
|------|---------|------|
| name | 是 | 模型中维度的唯一名称，通常与数据库列名对应 |
| description | 是 | 维度的详细业务描述，必须提供 |
| type | 是 | 底层数据库或数据仓库中的原始数据类型（如VARCHAR、INT、TIMESTAMP） |
| searchable | 否 | 指定是否为此维度创建搜索索引，默认为false |
| options | 否 | 预定义的可选值列表，适用于分类型数据 |

## 度量

度量是可聚合的定量字段，用于计算业务指标，如金额总和或数量统计。

| 字段 | 是否必需 | 描述 |
|------|---------|------|
| name | 是 | 模型中度量的唯一名称，通常与数据库列名对应 |
| description | 是 | 度量的详细业务描述，必须提供 |
| type | 是 | 数据库/仓库中定义的底层列类型，支持的聚合类型取决于查询引擎能力 |

## 最佳实践

- **描述性命名**：使用清晰、符合业务语境的名称，直观传达字段含义
- **详细描述**：提供完整描述，帮助用户理解字段的业务上下文和用途
- **明确数据类型**：准确设置类型，确保系统正确处理和显示数据值
- **标注单位**：在描述中注明度量单位（如"美元"、"天数"），避免误解

## 实例讲解

以下是一个标准化的订单语义模型示例，展示了维度和度量的实际应用：

```yaml
name: orders
description: >
  表示客户订单交易数据。包含每个订单的详细信息，
  包括唯一标识、客户关联、时间信息、状态和财务数值。
  这是基于事件的模型，对销售分析、收入跟踪和购买
  模式洞察至关重要。
dimensions:
  - name: order_id
    description: >
      订单交易的唯一标识符，作为此模型的主键。
      用于查询特定订单或计算不重复订单数。
    type: string
  - name: customer_id
    description: >
      关联到'customers'模型中'id'字段的外键，将订单
      链接到下单客户。是客户分析的关键连接点。
    type: string
  - name: order_date
    description: >
      订单创建日期。用于时间序列分析、按时间范围筛选
      （如日/周/月收入）及基于时间的队列分析。
    type: date
    searchable: true # 常用于日期筛选/搜索
  - name: status
    description: >
      订单的当前履行或支付状态（如'待处理'、'已发货'、
      '已送达'、'已取消'）。用于运营报告和状态筛选。
    type: string
    options: ["pending", "shipped", "delivered", "cancelled"] # 预定义值示例
measures:
  - name: amount
    description: >
      订单的总金额，通常以特定货币单位（如美元）计算。
      是收入相关计算（求和、平均值）的核心指标。
    type: number
  - name: item_count
    description: 订单中包含的商品总数，用于分析订单规模。
    type: integer
```

## 示例解析

### 维度:
- **order_id**：典型维度，可能为数字或字符串，但主要用于标识而非计算
- **customer_id**：典型维度，用于关联客户信息，而非直接参与计算
- **order_date**：时间类维度，提供事件发生的时间背景，是时序分析的基础
- **status**：分类文本维度，描述订单状态，options列表确保数据规范性

### 度量:
- **amount**：核心财务度量，表示订单金额，常用于汇总计算
- **item_count**：数量类度量，表示订单规模，用于各类聚合分析

这种结构清晰区分了用于上下文和筛选的描述性属性（维度）与用于计算和聚合的可量化值（度量），使模型既符合业务含义又便于分析使用。

## 后续步骤

完成维度和度量定义后，您可以：

1. 基于这些基础元素创建更复杂的业务计算指标
2. 建立模型间的关联关系，构建完整数据模型网络 


# 数据集/模型

## 模型的目的

语义模型在数据仓库之上提供了一个灵活的分析层，旨在创建业务流程和概念的数字孪生。它允许您在集中治理的位置定义业务逻辑、指标和关系。这种被称为"左移"的实践，将分析逻辑从下游BI工具或临时查询(如AI代理生成的查询)转移到这个上游建模层。通过这种方式，您可以确保逻辑被版本化、可测试，并在所有使用场景中一致应用。

## 模型结构

模型代表业务实体或概念，通常映射到数据库中的表或视图。每个模型定义包含名称，并可选择性地包含描述、维度、度量、指标、过滤器和实体(关系)。

### 必需字段

- **name**: 模型的唯一标识符，通常与数据库中的表名匹配
- **description**: 对模型代表的业务概念的清晰解释
- 至少需要定义dimensions(维度)、measures(度量)或metrics(指标)中的一种

### 可选字段

- **dimensions**: 维度定义
- **measures**: 度量定义
- **metrics**: 指标定义
- **filters**: 过滤器定义
- **relationships**: 关系定义

## 模型类型

### 1. 实体中心模型

代表核心业务对象或概念，通常映射到传统数据仓库中的维度表。

**特点**:
- 主要包含描述性维度(如客户名称、产品类别)
- 可能包含与实体直接相关的度量(如客户的信用额度)
- 通常包括基于实体属性的过滤器(如是否是高级客户)

**示例**: 客户、产品、员工、商店、发票

### 2. 事件中心模型

代表随时间发生的业务事件或交互，通常映射到事实表或事件流数据。

**特点**:
- 几乎总是包含时间戳维度
- 包含描述事件的维度(如页面URL、订单状态)
- 通常包含量化事件的度量(如订单金额、浏览时长)
- 通常与多个实体中心模型建立关系

**示例**: 页面浏览、订单、交易、登录事件、支持工单

### 3. 指标中心模型

围绕核心业务指标或KPI构建，通常对应于星型或雪花模式中的事实表。

**特点**:
- 主要包含构成核心指标的度量
- 维度通常是外键(如日期ID、客户ID)
- 主要目的是定义一个或多个重要指标(如总收入、活跃用户)
- 严重依赖关系连接回各自的维度/实体模型

**示例**: 月度收入、每日用户活动、营销活动表现

## 最佳实践

1. **从实体和事件开始**：大多数语义层主要围绕实体中心和事件中心模型构建
2. **明确关系**：清晰定义模型间关系
3. **混合使用**：现实数据通常需要组合使用不同模型类型
4. **保持专注**：每个模型应代表一个明确定义的概念
5. **性能考虑**：对于大型数据集，考虑使用指标中心模型进行预聚合

## 模型文件组织

- 每个模型定义必须放在单独的YAML文件中
- 建议按业务领域分组到子目录(如models/finance/, models/marketing/)
- 使用蛇形命名法(snake_case)命名模型
- 名称应反映业务概念而非技术表名

## 命名约定

- 使用蛇形命名法(如order_items)
- 选择反映业务实体或概念的名称
- 在整个语义层中保持命名模式一致

## 其他最佳实践

- **清晰的描述**：始终包含解释模型业务意义的描述文本
- **从简单开始**：从核心模型定义开始，根据需要逐步添加维度、度量等
- **与数据源对齐**：概念上应与数据库表或视图对齐，但不是严格要求
- **逻辑分组**：使用单独文件和子目录对相关模型进行逻辑分组 



# 过滤器定义

学习如何在语义模型中定义可重用的过滤器

## 过滤器介绍

过滤器定义了可命名的、可重复使用的条件，这些条件解析为真或假。它们封装了常见的查询约束，使用户能够轻松应用预定义的条件而无需重复编写逻辑。可以将它们视为保存的片段或常见的WHERE子句。

使用过滤器可以使查询更简洁、更一致、更易于理解，将分析重点放在特定问题上而不是样板过滤逻辑上。

基本结构定义如下：

```yaml
filters:
  - name: filter_name          # 必需: 唯一标识符(snake_case)
    description: "业务定义" # 必需: 清晰的解释
    expr: "SQL布尔表达式" # 可选(如果名称匹配布尔维度)
    args:                      # 可选: 用于参数化过滤器
      - name: param_name
        type: param_type
        description: "参数描述" # 必需
        default: default_value        # 可选
```

## 过滤器 vs 维度/度量

区分过滤器与维度和度量至关重要：

- **维度**：描述属性(如国家、状态)。用于分组、切片和直接过滤
- **度量**：可量化、可聚合的值。用于聚合函数和HAVING子句
- **过滤器**：定义可重复使用的条件或子集。必须解析为布尔值并生成WHERE子句逻辑

**指导原则**：
- 如果布尔概念描述直接属性(如用户存储的活动状态)，使用布尔维度
- 如果需要可重复使用的条件(特别是涉及计算、日期逻辑、组合属性或相关数据)，使用过滤器

## 定义基础过滤器

基础过滤器定义简单的非参数化条件。

| 字段 | 是否必需 | 描述 |
|------|---------|------|
| name | 是 | 模型中过滤器的唯一名称 |
| description | 是 | 解释过滤器代表的条件 |
| expr | 否 | 定义过滤器逻辑的SQL表达式(必须评估为布尔值) |
| args | 否 | 参数化过滤器表达式的参数对象列表 |

## 示例：基础过滤器

```yaml
filters:
  # 直接映射到布尔维度的过滤器(省略expr)
  - name: is_active
    description: 当前标记为活动的用户。

  # 基于单一维度表达式的过滤器
  - name: is_us_based
    description: 位于美国的用户。
    expr: "country = 'USA'"

  # 使用基于时间表达式的过滤器
  - name: recent_signups
    description: 最近90天内注册的用户。
    expr: "created_at >= DATE_SUB(CURRENT_DATE(), INTERVAL 90 DAY)"
```

## 高级过滤器类型

过滤器可以包含参数或引用相关模型中的数据。

### 参数化过滤器

创建通过args接受用户输入的动态过滤器。使用双花括号{{parameter_name}}引用参数。

#### 示例：参数化过滤器

```yaml
filters:
  - name: orders_within_date_range
    description: 选择在特定日期范围内(含)下单的订单。
    expr: "orders.order_date BETWEEN {{start_date}} AND {{end_date}}"
    args:
      - name: start_date
        type: date
        description: "过滤器范围的开始日期(YYYY-MM-DD)"
      - name: end_date
        type: date
        description: "过滤器范围的结束日期(YYYY-MM-DD)"
```

### 跨模型过滤器

过滤器可以基于相关模型中的数据定义条件，前提是存在关系。语义层使用这些关系在应用过滤器的查询中生成必要的连接或子查询。

#### 示例：跨模型过滤器

```yaml
filters:
  - name: customers_in_region_managed_by
    description: 选择位于分配给特定销售经理的区域的客户。
    expr: "region.sales_manager = {{manager_name}}"
    args:
      - name: manager_name
        type: string
        description: "要过滤的销售经理姓名"
```

## 最佳实践与注意事项

遵循这些指南以定义有效的过滤器。

### 通用最佳实践

- **确保布尔输出**：expr必须评估为布尔值(真/假)
- **清晰命名与描述**：使用描述性名称和清晰的描述
- **匹配布尔维度**：对于表示dimension = true的过滤器，名称应与布尔维度相同并省略expr
- **记录依赖关系**：明确说明对相关模型数据或参数的依赖

### 表达式最佳实践

- **使用表前缀**：强烈建议当expr涉及可能存在于其他模型中的字段时
- **清晰性和可读性**：编写清晰的expr逻辑
- **空值处理**：注意空值处理

### 良好表达式示例(参数化)：

```yaml
filters:
  - name: users_signed_up_between
    description: 基于注册日期范围过滤用户。
    expr: "users.created_at BETWEEN '{{start_date}}' AND '{{end_date}}'"
    args:
      - name: start_date
        type: date
        description: 开始日期(YYYY-MM-DD)
      - name: end_date
        type: date
        description: 结束日期(YYYY-MM-DD)
```

### 参数最佳实践

- **显式类型**：始终为每个参数定义类型
- **参数字段**：确保args列表中的每个参数包含必需字段
- **默认值**：为可选参数提供默认值
- **清晰文档**：为每个参数提供有意义的描述

## 综合示例

以下是客户模型上下文中的示例，包含各种过滤器类型和实践：

```yaml
name: customers
description: >
  表示单个客户实体。包含联系信息、注册详情、
  位置和衍生指标如生命周期价值。
dimensions:
  - name: id
    description: 客户的唯一标识符。
    type: string
  - name: country
    description: 客户所在国家。
    type: string
  - name: is_active
    description: 布尔标志，指示客户账户当前是否被视为活动状态。
    type: boolean
relationships:
  - name: orders
    description: 将此客户链接到'orders'模型中所有相关的订单记录。
filters:
  # 基础过滤器
  - name: is_active
    description: 当前标记为活动的客户。

  # 参数化过滤器
  - name: registered_after_date
    description: 在指定日期或之后注册的客户。
    expr: "customers.user_created_at >= '{{signup_date}}'"
    args:
      - name: signup_date
        type: date
        description: 最小注册日期(YYYY-MM-DD)

  # 跨模型过滤器
  - name: ordered_sku_recently
    description: "最近N天内订购特定SKU的客户。需要通过'orders'关系连接。"
    expr: "orders.product_sku = '{{sku}}' AND orders.order_date >= DATE_SUB(CURRENT_DATE(), INTERVAL {{days}} DAY)"
    args:
      - name: sku
        type: string
        description: 要检查的产品SKU
      - name: days
        type: number
        description: 回溯天数
        default: 30
```

# 什么是语义层？

关于语义层的定义存在许多争论。在OraAI中，我们所说的语义层是指一组YAML文件，用于定义以下内容：

- **模型（Models）**：需要分析的核心业务对象。例如订单、产品、客户等。
- **维度（Dimensions）**：通常是非数字属性，用于数据分组、过滤或分段。例如日期、类别、状态、名称等。
- **度量（Measures）**：可量化的属性。例如产品成本、会话时长、温度等。
- **指标（Metrics）**：基于度量和维度的计算与业务逻辑。例如总收入、平均订单价值等。
- **过滤器（Filters）**：命名的布尔条件，用于常见查询约束。例如is_active（是否活跃）、is_deleted（是否删除）、is_paid（是否支付）等。
- **关系（Relationships）**：不同模型之间的关联。例如客户模型与订单模型的连接。

这些定义使OraAI能够：
- 生成准确的SQL语句
- 统一理解业务指标
- 从数据中提供可靠的洞察

与其他语义层不同，OraAI的语义层并非抽象的计算层，而更接近数据目录（data catalog）。我们通过YAML配置帮助AI代理理解数据并构建可靠查询。

## 为什么使用语义层？

### 1. 集中管理业务逻辑
- 在版本化文件中定义业务逻辑和指标，确保所有分析中一致复用。

### 2. 明确数据关系
- 映射表间关联，使OraAI能高效导航数据模型。

### 3. 补充上下文与含义
- 通过描述和元数据，帮助AI理解数据的业务意义。

## 关键点梳理

| 术语 | 作用 | 示例 |
|------|------|------|
| 模型 | 核心业务对象 | 订单、客户、产品 |
| 维度 | 非数字属性，用于分组/过滤 | 日期、状态、类别 |
| 度量 | 可量化数值 | 成本、时长、温度 |
| 指标 | 基于度量+维度的计算逻辑 | 总收入、平均订单价值 |
| 过滤器 | 预定义的查询条件 | is_active=true（活跃用户） |
| 关系 | 连接不同模型的字段 | 客户ID → 订单表中的客户字段 |

## 优势总结：

- **一致性**：避免重复定义指标，确保分析结果统一。
- **可读性**：YAML文件结构化，便于团队协作维护。
- **AI友好**：为AI提供数据上下文，减少查询错误。 


# 指标定义

在语义层中定义计算和关键绩效指标(KPIs)

## 指标介绍

指标是基于语义模型中度量或维度的计算。它们代表关键业务绩效指标(KPIs)和其他常用计算，集中封装业务逻辑。

基本结构定义如下：

```yaml
metrics:
  - name: metric_name     # 必需 - 指标的唯一标识符
    expr: "SQL_expression" # 必需 - 定义计算的SQL表达式
    description: "业务定义" # 必需 - 指标的清晰解释
    args:                   # 可选 - 用于参数化指标
      - name: parameter_name
        type: parameter_type
        description: "参数描述"
        default: default_value        # 可选
```

## 指标 vs 度量

区分指标(metrics)与度量(measures)和维度(dimensions)很有帮助：

- **维度**：用于分组或筛选的描述性属性(如国家、状态)
- **度量**：表示对单列的直接可聚合计算，通常对应于原始数据字段(如SUM(amount)定义为名为total_amount的度量)
- **指标**：定义更复杂的计算、业务逻辑或关键绩效指标(KPIs)，通常组合多个度量、维度或涉及条件逻辑

**指导原则**：
- 如果需要简单的直接列聚合(如求和、计数、平均)，定义为度量
- 如果需要定义特定的业务KPI、比率、涉及多个步骤或度量的计算，或参数化计算，定义为指标

## 定义基础指标

核心指标包含名称、计算表达式(expr)和描述。

| 字段 | 是否必需 | 描述 |
|------|---------|------|
| name | 是 | 模型中指标的唯一名称 |
| expr | 是 | 定义指标计算的SQL表达式，可引用度量、维度和参数 |
| description | 是 | 指标的人类可读描述 |
| args | 否 | 参数化指标表达式的参数对象列表 |

## 编写表达式

expr字段使用与数据仓库兼容的SQL语法定义指标计算。

- **引用字段**：可直接引用同一模型中定义的维度和度量
- **聚合函数**：使用标准SQL聚合函数(SUM, AVG, COUNT等)
- **运算符**：支持算术、逻辑和比较运算符
- **函数**：可使用大多数标准SQL函数

## 示例：基础指标

```yaml
metrics:
  - name: total_revenue
    expr: "SUM(orders.amount)"
    description: >
      通过汇总查询上下文中所有订单的'amount'计算总收入。代表毛收入。

  - name: order_count
    expr: "COUNT(DISTINCT orders.order_id)"
    description: >
      基于'order_id'的不同计数，查询上下文中包含的唯一订单总数。
```

## 高级指标类型

除了基础计算，还可以创建更动态和互连的指标。

### 参数化指标

创建接受用户查询时输入的灵活指标。使用双花括号{{parameter_name}}引用参数。

#### 示例：参数化指标

```yaml
metrics:
  - name: revenue_in_last_n_days
    description: 计算用户指定或默认为30天的最近N天的总收入。
    expr: "SUM(CASE WHEN orders.order_date >= DATE_SUB(CURRENT_DATE(), INTERVAL {{days}} DAY) THEN orders.amount ELSE 0 END)"
    args:
      - name: days
        type: number
        description: 收入计算中包含的天数。
        default: 30
```

### 跨模型指标

使用点符号(related_model_name.field_name)引用相关模型中的字段。这需要在相关模型之间定义关系。

#### 示例：跨模型指标

```yaml
metrics:
  - name: revenue_per_customer
    description: >
      订单查询上下文中每个唯一客户的平均收入。需要通过'customer'关系将'orders'模型与'customers'模型连接。
    expr: "SUM(orders.amount) / NULLIF(COUNT(DISTINCT customer.id), 0)"
```

## 最佳实践与注意事项

遵循这些指南创建健壮、可理解和可维护的指标。

### 通用最佳实践

- **业务导向**：定义代表有意义业务KPI的指标
- **清晰命名与描述**：使用一致、描述性名称并提供清晰描述
- **利用现有组件**：通过引用模型中现有度量和维度构建指标
- **记录依赖关系**：明确说明指标对参数或相关模型数据的依赖

### 表达式最佳实践

- **类型一致性**：确保表达式解析为适合聚合的适当数据类型
- **使用表前缀**：引用相关模型字段时始终使用表限定名称
- **SQL约定**：
  - 用括号包裹复杂表达式确保正确运算符优先级
  - 使用NULLIF处理除零错误
  - 注意空值处理
- **避免循环引用**

### 良好表达式示例(参数化)：

```yaml
metrics:
  - name: avg_order_value_by_date
    description: 指定日期范围内订单的平均订单价值。
    expr: "SUM(CASE WHEN orders.created_at BETWEEN '{{start_date}}' AND '{{end_date}}' THEN orders.amount ELSE 0 END) / NULLIF(COUNT(CASE WHEN orders.created_at BETWEEN '{{start_date}}' AND '{{end_date}}' THEN orders.id ELSE NULL END), 0)"
    args:
      - name: start_date
        type: date
        description: 开始日期(YYYY-MM-DD)。
      - name: end_date
        type: date
        description: 结束日期(YYYY-MM-DD)。
```

### 参数最佳实践

- **显式类型**：始终为每个参数定义类型
- **参数字段**：确保args列表中的每个参数包含必需字段
- **默认值**：为可选参数提供默认值
- **清晰文档**：为每个参数提供有意义的描述
- **占位符**：使用双花括号引用参数

## 综合示例

以下是订单模型上下文中的示例，包含各种指标类型和实践：

```yaml
name: orders
description: >
  表示单个客户订单交易。包含每个订单的详细信息，
  包括标识符、客户链接、时间、状态和财务价值。
dimensions:
  - name: order_id
    description: 订单交易的唯一标识符。
    type: string
  - name: order_date
    description: 订单下单日期。
    type: date
  - name: status
    description: 订单的当前履行或支付状态。
    type: string
  - name: customer_id
    description: 将订单链接到客户的标识符。
    type: string
measures:
  - name: amount
    description: 订单的总货币价值(如美元)。
    type: number
  - name: item_count
    description: 订单中包含的不同商品数量。
    type: integer
relationships:
  - name: customer
    description: 将订单与其对应的客户链接。
metrics:
  # 基础指标
  - name: total_revenue
    description: 通过汇总查询上下文中所有订单的'amount'计算总收入。
    expr: "SUM(orders.amount)"

  - name: average_order_value
    description: 计算查询结果中所有订单的平均'amount'。
    expr: "AVG(orders.amount)"

  # 参数化指标
  - name: revenue_by_period
    description: 最近N天内下单的订单总收入。
    expr: "SUM(CASE WHEN orders.order_date >= DATE_SUB(CURRENT_DATE(), INTERVAL {{days}} DAY) THEN orders.amount ELSE 0 END)"
    args:
      - name: days
        type: number
        description: 计算中回溯的天数(整数)。
        default: 30

  # 跨模型指标
  - name: revenue_per_customer
    description: 订单查询上下文中每个唯一客户的平均收入。
    expr: "SUM(orders.amount) / NULLIF(COUNT(DISTINCT customer.id), 0)"

  # 更复杂的参数化指标
  - name: revenue_growth_vs_prior_periods
    description: >
      计算与N个周期前相比的总收入百分比增长。
      注意：此计算依赖于`LAG`窗口函数。
    expr: "(SUM(orders.amount) - LAG(SUM(orders.amount), {{periods}}) OVER (ORDER BY orders.order_date)) / NULLIF(LAG(SUM(orders.amount), {{periods}}) OVER (ORDER BY orders.order_date), 0)"
    args:
      - name: periods
        type: number
        description: 比较的先前周期数(如1表示紧接的前一个周期)。
        default: 1
``` 


# 工作原理

## 数据目录基础

OraAI的核心是AI原生数据目录，这是系统理解数据环境的智能知识库：

### 目录组成：
- **模型**：主要来自dbt等数据建模工具，代表结构化表和视图数据
- **语义定义**：通过YAML文件为原始模型添加业务上下文和语义含义，包括：
  - 维度和度量
  - 指标
  - 过滤器
  - 枚举值(字段有效值)
  - 其他常见语义层组件
- **部署方式**：使用CLI工具将数据目录部署到OraAI平台，通常在管理数据模型的代码库(如dbt项目目录)中运行

## 用户问题解答流程

### 1. 智能搜索：
- 执行语义搜索而非简单关键词匹配
- 模仿人类分析师直觉搜索相关表、列、指标和定义

### 2. 规划阶段：
- **分析与澄清**：系统会结合已检索到的上下文，对用户的请求进行深入分析，并在必要时主动提出澄清性问题，确保理解用户真实意图。
- **计划制定**：基于分析结果，系统会制定详细的分步执行计划（内部“待办清单”），例如：
  - “利用维度Y和度量Z来构建新的业务指标”
  - “新建仪表板并添加多个关键指标，如销售额、用户增长率、活跃用户数等”

## 执行循环

### 1. 迭代创建：
- **生成数据产物**：
  - 可视化(指标)：配置好的可视化或特定数据计算
  - 仪表板和报告：展示生成的指标和洞察
- **安全语义查询生成**：
  - 智能解析SQL
  - 利用语义层定义确保正确的表连接
  - 注入预定义表达式(如自定义维度或度量)
  - 包含安全防护措施和访问控制

### 2. 版本控制产物：
- 以文件形式创建指标和仪表板
- 可轻松集成到Git等版本控制系统

### 3. 审查与进展：
- 执行过程中包含中间审查步骤
- 检查待办清单，标记已完成任务

### 4. 完成：
- 持续创建和审查循环，直到完成所有计划任务 


